<?php

/*
 * This file is part of the Serenity package.
 *
 * (c) CP Creation <web@cpcreation.fr>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Serenity\SaasBundle\Entity;

use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Serenity\CoreBundle\Doctrine\Contract\DisabledInterface;
use Serenity\CoreBundle\Trait\DisabledTrait;
use Serenity\CoreBundle\Trait\IdentifierTrait;
use Serenity\CoreBundle\Trait\SoftDeletableTrait;
use Serenity\CoreBundle\Trait\TimestampTrait;
use Serenity\SaasBundle\Doctrine\Repository\ProjectRepository;
use Symfony\Component\Validator\Constraints as Assert;

#[ORM\Table(name: 'serenity_saas_project')]
#[ORM\Entity(repositoryClass: ProjectRepository::class)]
#[ORM\HasLifecycleCallbacks]
#[Gedmo\SoftDeleteable(fieldName: 'deletedAt', timeAware: false, hardDelete: true)]
class Project implements DisabledInterface
{
    use DisabledTrait;
    use IdentifierTrait {
        IdentifierTrait::__construct as private identifierConstruct;
    }
    use SoftDeletableTrait;
    use TimestampTrait;

    #[Assert\NotBlank()]
    #[Assert\Length(min: 3, max: 255)]
    #[ORM\Column(type: 'string', length: 255)]
    private string $name;

    #[ORM\Column(type: 'string', length: 255, unique: true)]
    private string $identifier;

    #[Assert\NotBlank()]
    #[Assert\Length(min: 3, max: 255)]
    #[ORM\Column(type: 'string', length: 255, unique: true)]
    private string $domainIn;

    #[Assert\Length(min: 3, max: 255)]
    #[ORM\Column(type: 'string', length: 255, unique: true, nullable: true)]
    private ?string $domainOut = null;

    #[ORM\Column(type: 'datetime', nullable: true)]
    private ?\DateTime $checkDNS = null;

    #[ORM\ManyToOne(targetEntity: Tenant::class, inversedBy: 'projects')]
    private Tenant $tenant;

    public function __construct()
    {
        $this->identifierConstruct();
        $timestamp = base_convert((string) time(), 10, 36);
        $randomNumber = (string) random_int(100000, 999999);
        $randomPart = mb_substr(base_convert($randomNumber, 10, 36), 0, 4);
        $this->identifier = "{$timestamp}-{$randomPart}";
    }

    public function getName(): string
    {
        return $this->name;
    }

    public function setName(string $name): void
    {
        $this->name = $name;
    }

    public function getIdentifier(): string
    {
        return $this->identifier;
    }

    public function setManualIdentifier(string $identifier): void
    {
        $this->identifier = $identifier;
    }

    public function getDomainIn(): string
    {
        return $this->domainIn;
    }

    public function setDomainIn(string $domainIn): void
    {
        $this->domainIn = $domainIn;
    }

    public function getDomainOut(): ?string
    {
        return $this->domainOut;
    }

    public function setDomainOut(?string $domainOut): void
    {
        $this->domainOut = $domainOut;
    }

    public function getCheckDNS(): ?\DateTime
    {
        return $this->checkDNS;
    }

    public function setCheckDNS(?\DateTime $checkDNS): void
    {
        $this->checkDNS = $checkDNS;
    }

    public function getTenant(): Tenant
    {
        return $this->tenant;
    }

    public function setTenant(Tenant $tenant): void
    {
        $this->tenant = $tenant;
    }
}
