<?php

/*
 * This file is part of the Serenity package.
 *
 * (c) CP Creation <web@cpcreation.fr>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Serenity\MediaBundle\Trait;

use Serenity\CoreBundle\Contract\ConfigInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\PropertyAccess\PropertyAccess;
use Symfony\Contracts\Service\Attribute\Required;
use Symfony\UX\LiveComponent\Attribute\LiveArg;
use Symfony\UX\LiveComponent\Attribute\LiveListener;
use Symfony\UX\LiveComponent\Attribute\LiveProp;

trait ComponentMediaTrait
{
    private string $relativeDefaultLocalDirectory;
    private string $publicLocalDirectory;

    #[Required]
    public function setLocalDirectory(
        #[Autowire(param: 'relative_default_local_directory')]
        string $relativeDefaultLocalDirectory,
        #[Autowire(param: 'public_local_directory')]
        string $publicLocalDirectory,
    ): void {
        $this->relativeDefaultLocalDirectory = $relativeDefaultLocalDirectory;
        $this->publicLocalDirectory = $publicLocalDirectory;
    }

    #[LiveProp(writable: false)]
    public array $uploadedFiles = [];

    #[LiveListener('uploadedFilesChanged')]
    public function changeUploadedFiles(#[LiveArg] array $addUploadedFiles = [], #[LiveArg] array $removeUploadedFiles = []): void
    {
        foreach ($addUploadedFiles as $key => $uploadedFile) {
            $this->uploadedFiles[preg_replace('/^[^_]+_/', '', $key)] = $uploadedFile;
        }
        foreach ($removeUploadedFiles as $uploadedFileKey) {
            unset($this->uploadedFiles[preg_replace('/^[^_]+_/', '', $uploadedFileKey)]);
        }
        $this->formValues = array_replace_recursive($this->formValues, $this->deserializeUploadedFiles($this->uploadedFiles));
    }

    protected function deserializeUploadedFiles(array $flatArray): array
    {
        $nested = [];

        foreach ($flatArray as $flatKey => $value) {
            $keys = explode('_', $flatKey);
            $ref = &$nested;

            foreach ($keys as $key) {
                if (!isset($ref[$key]) || !\is_array($ref[$key])) {
                    $ref[$key] = [];
                }
                $ref = &$ref[$key];
            }

            if (\is_array($value) && isset($value['path'], $value['originalName'], $value['mimeType'])) {
                $value = new UploadedFile(
                    $this->publicLocalDirectory.'/'.$value['path'], // chemin réel du fichier dans /tmp
                    $value['originalName'],
                    $value['mimeType'],
                    null,
                    true  // $test = true : ce n’est pas un vrai upload HTTP (mais simulé)
                );
            }
            $ref = $value;
        }

        return $nested;
    }

    /** Utile pour persister manuellement si besoins les entités se touvant dans un json_document comme les Attachments ( Image, File, .. )   */
    protected function manualPersistAttachment(object $object, array $deserializeUploadedFiles): object
    {
        foreach ($this->findKeysLeadingToUploadedFile($deserializeUploadedFiles) as $property) {
            $accessor = PropertyAccess::createPropertyAccessor();
            $value = $accessor->getValue($object, $property);
            if ($object instanceof ConfigInterface) {
                $this->entityManager->persist($value);
                $this->entityManager->flush();
            }
        }

        return $object;
    }

    private function findKeysLeadingToUploadedFile(array $data, array $parents = []): array
    {
        $results = [];

        foreach ($data as $key => $value) {
            $currentPath = [...$parents, $key];

            if ($value instanceof UploadedFile) {
                // On veut la clé juste avant UploadedFile (ex: "logo")
                if (\count($currentPath) >= 2) {
                    $results[] = $currentPath[\count($currentPath) - 2];
                }
            } elseif (\is_array($value)) {
                $results = array_merge(
                    $results,
                    $this->findKeysLeadingToUploadedFile($value, $currentPath)
                );
            }
        }

        return array_unique($results);
    }
}
