<?php

/*
 * This file is part of the Serenity package.
 *
 * (c) CP Creation <web@cpcreation.fr>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Serenity\MediaBundle\Doctrine\EventListener;

use Doctrine\Bundle\DoctrineBundle\Attribute\AsDoctrineListener;
use Doctrine\ORM\Event\PreFlushEventArgs;
use Doctrine\ORM\Events;
use Doctrine\ORM\UnitOfWork;
use Serenity\MediaBundle\Attribute\Attachment as AttachmentAttribute;
use Serenity\MediaBundle\Attribute\AttachmentCollection;
use Serenity\MediaBundle\Contract\AttachmentInterface;
use Serenity\MediaBundle\Entity\Attachment;

#[AsDoctrineListener(event: Events::preFlush)]
class MediaParentListener
{
    public function preFlush(PreFlushEventArgs $args): void
    {
        $entityManager = $args->getObjectManager();
        $unitOfWork = $entityManager->getUnitOfWork();

        foreach ($unitOfWork->getScheduledEntityInsertions() as $entity) {
            if ($entity instanceof AttachmentInterface) {
                $this->setParentForImages($entity, $unitOfWork);
            }
        }
    }

    private function setParentForImages(AttachmentInterface $parent, UnitOfWork $unitOfWork): void
    {
        $reflectionClass = new \ReflectionClass($parent);
        foreach ($reflectionClass->getProperties() as $property) {
            if ($property->getAttributes(AttachmentCollection::class)) {
                $value = $property->getValue($parent);
                if (is_iterable($value)) {
                    foreach ($value as $item) {
                        if ($item instanceof Attachment) {
                            $this->setParentIdForAttachment($item, $parent, $unitOfWork);
                        }
                    }
                }
            }
            if ($property->getAttributes(AttachmentAttribute::class)) {
                $value = $property->getValue($parent);
                if ($value instanceof Attachment) {
                    $this->setParentIdForAttachment($value, $parent, $unitOfWork);
                }
            }
        }
    }

    private function setParentIdForAttachment(
        Attachment $attachment,
        AttachmentInterface $parent,
        UnitOfWork $unitOfWork,
    ): void {
        $attachment->setParentId($parent->getId());

        $unitOfWork->recomputeSingleEntityChangeSet(
            $unitOfWork->getEntityPersister($attachment::class)->getClassMetadata(),
            $attachment
        );
    }
}
