<?php

/*
 * This file is part of the Serenity package.
 *
 * (c) CP Creation <web@cpcreation.fr>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Serenity\CoreBundle\Entity;

use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Serenity\CoreBundle\Aggregate\AggregateRoot;
use Serenity\CoreBundle\Aggregate\AggregateRootInterface;
use Serenity\CoreBundle\Contract\ConfigInterface;
use Serenity\CoreBundle\Contract\SerenitySaasInterface;
use Serenity\CoreBundle\Doctrine\Repository\ConfigRepository;
use Serenity\CoreBundle\Trait\IdentifierTrait;
use Serenity\CoreBundle\Trait\SerenitySaasTrait;
use Serenity\CoreBundle\Trait\TimestampTrait;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Validator\Constraints as Assert;

#[ORM\Entity(repositoryClass: ConfigRepository::class)]
#[ORM\Table(name: 'serenity_core_config')]
#[ORM\UniqueConstraint(name: 'unique_config_per_project', columns: ['name', 'saasIdentifier'], options: ['nulls_not_distinct' => true])]
#[UniqueEntity(fields: ['name', 'saasIdentifier'], message: 'entity.unique.config')]
#[ORM\HasLifecycleCallbacks]
class Config implements AggregateRootInterface, SerenitySaasInterface
{
    use AggregateRoot;
    use IdentifierTrait {
        IdentifierTrait::__construct as private identifierConstruct;
    }

    use SerenitySaasTrait;
    use TimestampTrait;

    #[ORM\Column(type: Types::STRING, length: 255)]
    private string $name;

    #[Assert\Valid]
    #[ORM\Column(type: 'json_document', options: ['jsonb' => true])]
    private ConfigInterface $value;

    public function getName(): string
    {
        return $this->name;
    }

    public function getValue(): ConfigInterface
    {
        return $this->value;
    }

    public function setValue(ConfigInterface $value): void
    {
        $this->value = $value;
        $this->name = $this->value->getName();
        $this->saasIdentifierAutoGenerated = $this->value->isSaasIdentifierConfig();
    }

    public function setSaasIdentifier(?string $saasIdentifier = null): void
    {
        $this->saasIdentifier = $saasIdentifier;
        if (!$this->value->isSaasIdentifierConfig()) {
            $this->saasIdentifier = null;
        }
    }

    public function isSaasIdentifierAutoGenerated(): bool
    {
        return $this->saasIdentifierAutoGenerated;
    }
}
