<?php

declare(strict_types=1);

/*
 * This file is part of the Serenity package.
 *
 * (c) CP Creation <web@cpcreation.fr>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Serenity\CoreBundle\Twig\Extension;

use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

class EntityExtension extends AbstractExtension
{
    public function __construct(
        private readonly ManagerRegistry $doctrine,
    ) {
    }

    public function getFunctions(): array
    {
        return [
            new TwigFunction('entity_data', $this->getEntityData(...)),
        ];
    }

    /**
     * @param class-string $entityClass
     * @param list<string> $fields
     *
     * @return array<string, mixed>
     *
     * Récupère certains champs d'une entité (supporte les relations imbriquées)
     */
    public function getEntityData(string $entityClass, mixed $id, array $fields): array
    {
        $em = $this->doctrine->getManagerForClass($entityClass);

        if (!$em instanceof EntityManagerInterface) {
            throw new \InvalidArgumentException(\sprintf('L’EntityManager pour %s n’est pas une instance de Doctrine\ORM\EntityManagerInterface', $entityClass));
        }

        $qb = $em->createQueryBuilder()
            ->from($entityClass, 'e')
            ->where('e.id = :id')
            ->setParameter('id', $id)
            ->setMaxResults(1);

        $selects = [];
        $joins = [];

        foreach ($fields as $field) {
            $parts = explode('.', $field);
            $alias = 'e';
            $path = [];

            // On traverse chaque niveau de relation
            for ($i = 0; $i < \count($parts) - 1; ++$i) {
                $relation = $parts[$i];
                $path[] = $relation;
                $joinKey = implode('_', $path); // ex: customer_address_country
                $joinAlias = 'j_'.$joinKey;

                if (!isset($joins[$joinKey])) {
                    $qb->leftJoin("$alias.$relation", $joinAlias);
                    $joins[$joinKey] = $joinAlias;
                }

                $alias = $joins[$joinKey];
            }

            // Le dernier élément est le champ demandé
            $finalField = $parts[\count($parts) - 1];
            $fieldAlias = implode('_', $parts); // ex: customer_address_city
            $selects[] = "$alias.$finalField AS $fieldAlias";
        }

        $qb->select(implode(', ', $selects));

        $result = $qb->getQuery()->getOneOrNullResult(AbstractQuery::HYDRATE_ARRAY);

        if (null === $result) {
            throw new NotFoundHttpException(\sprintf('%s #%s introuvable.', $entityClass, $id));
        }

        return $result;
    }
}
