<?php

/*
 * This file is part of the Serenity package.
 *
 * (c) CP Creation <web@cpcreation.fr>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Serenity\CoreBundle\Service;

use Doctrine\ORM\EntityManagerInterface;
use Knp\Menu\ItemInterface;
use Knp\Menu\Provider\MenuProviderInterface;
use Knp\Menu\Twig\Helper as MenuHelper;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\RequestStack;

class BreadcrumbBuilder
{
    public function __construct(
        private readonly MenuProviderInterface $menuProvider,
        #[Autowire(service: 'knp_menu.helper')]
        private readonly MenuHelper $menuHelper,
        private readonly RequestStack $requestStack,
        private EntityManagerInterface $em,
    ) {
    }

    /**
     * @return array<int, array{
     *     label: string,
     *     uri: string|null,
     *     entity?: object
     * }>
     */
    public function build(string $menuName = 'sidebarAdminMenu'): array
    {
        $request = $this->requestStack->getCurrentRequest();
        if (!$request) {
            return [];
        }

        $route = $request->attributes->get('_route');
        $params = $request->attributes->get('_route_params', []);
        $entityCls = $request->attributes->get('entity');

        // 1. Calculer la route de base (on enlève _edit/_new/_show si présent)
        $baseRoute = preg_replace('/_(edit|new|show)$/', '', $route);

        // 2. Récupérer le menu
        $menu = $this->menuProvider->get($menuName);

        // 3. Trouver l’item correspondant à la baseRoute
        $baseItem = $this->findItemByRoute($menu, $baseRoute);

        $breadcrumbs = $baseItem
            ? $this->menuHelper->getBreadcrumbsArray($baseItem)
            : [];

        // 4. Compléter avec infos dynamiques
        if ($entityCls) {
            $entity = null;
            if (isset($params['id'])) {
                $entity = $this->em->getRepository($entityCls)->find($params['id']);
            }

            if ($route && str_ends_with($route, '_new')) {
                $breadcrumbs[] = [
                    'label' => 'Nouveau',
                    'uri' => null,
                ];
            }

            if ($route && str_ends_with($route, '_edit') && $entity) {
                $breadcrumbs[] = [
                    'label' => \sprintf(
                        'Édition de  %s ',
                        method_exists($entity, '__toString')
                            ? (string) $entity
                            : (method_exists($entity, 'getId') ? $entity->getId() : '—')
                    ),
                    'uri' => null,
                    'entity' => $entity,
                ];
            }

            if ($route && str_ends_with($route, '_show') && $entity) {
                $breadcrumbs[] = [
                    'label' => \sprintf(
                        'Détails de  %s ',
                        method_exists($entity, '__toString')
                            ? (string) $entity
                            : (method_exists($entity, 'getId') ? $entity->getId() : '—')
                    ),
                    'uri' => null,
                    'entity' => $entity,
                ];
            }
        }

        // Retirer le premier élément si c’est "root"
        if (!empty($breadcrumbs) && 'root' === mb_strtolower($breadcrumbs[0]['label'])) {
            array_shift($breadcrumbs);
        }

        return $breadcrumbs;
    }

    private function findItemByRoute(ItemInterface $menu, string $route): ?ItemInterface
    {
        if ($menu->getExtra('routes')) {
            foreach ($menu->getExtra('routes') as $r) {
                if (($r['route'] ?? null) === $route) {
                    return $menu;
                }
            }
        }

        foreach ($menu->getChildren() as $child) {
            $found = $this->findItemByRoute($child, $route);
            if ($found) {
                return $found;
            }
        }

        return null;
    }
}
