<?php

declare(strict_types=1);

/*
 * This file is part of the Serenity package.
 *
 * (c) CP Creation <web@cpcreation.fr>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Serenity\CoreBundle\EventListener;

use Doctrine\Bundle\DoctrineBundle\Attribute\AsDoctrineListener;
use Doctrine\DBAL\Event\SchemaCreateTableColumnEventArgs;
use Doctrine\DBAL\Events;
use Doctrine\ORM\Tools\Event\GenerateSchemaTableEventArgs;
use Doctrine\ORM\Tools\ToolEvents;
use Linkage\DoctrineRowLevelSecurity\RowLevelSecurityConfig;
use Serenity\CoreBundle\Attribute\RowLevelSecurity;
use Serenity\CoreBundle\Factory\RowLevelSecuritySqlFactory;

#[AsDoctrineListener(event: ToolEvents::postGenerateSchemaTable)]
#[AsDoctrineListener(event: Events::onSchemaCreateTableColumn)]
class RowLevelSecurityListener
{
    private const string RLS_VISITED_OPTION_NAME = 'rowLevelSecurityConfigured';

    public function postGenerateSchemaTable(GenerateSchemaTableEventArgs $args): void
    {
        $reflClass = $args->getClassMetadata()->reflClass;

        if (null === $reflClass) {
            return;
        }

        $rlsAttributes = $reflClass->getAttributes(RowLevelSecurity::class);
        if (0 === \count($rlsAttributes)) {
            return;
        }

        $table = $args->getClassTable();
        $table->addOption(RowLevelSecurityConfig::RLS_OPTION_NAME, $rlsAttributes[array_key_first($rlsAttributes)]->getArguments());
    }

    public function onSchemaCreateTableColumn(SchemaCreateTableColumnEventArgs $args): void
    {
        if (
            !$args->getTable()->hasOption(RowLevelSecurityConfig::RLS_OPTION_NAME)
            || $args->getTable()->hasOption(self::RLS_VISITED_OPTION_NAME)
        ) {
            return;
        }

        $table = $args->getTable();
        $rlsOptions = $table->getOption(RowLevelSecurityConfig::RLS_OPTION_NAME);
        $policyName = $rlsOptions['name'] ?? \sprintf('%s_policy', $table->getName());
        $tableName = $table->getQuotedName($args->getPlatform());
        $roleName = $rlsOptions['role'];
        $using = $rlsOptions['using'];
        $for = $rlsOptions['for'];
        $withCheck = $rlsOptions['withCheck'];

        $sqlFactory = new RowLevelSecuritySqlFactory();

        // ⚡ Ajout du FORCE RLS
        foreach ($sqlFactory->createEnableSqls($policyName, $tableName, $roleName, $using, $for, $withCheck) as $sql) {
            $args->addSql($sql);
        }

        $table->addOption(self::RLS_VISITED_OPTION_NAME, true);
    }
}
