<?php

/*
 * This file is part of the Serenity package.
 *
 * (c) CP Creation <web@cpcreation.fr>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Serenity\CoreBundle\Doctrine\EventListener;

use Doctrine\Bundle\DoctrineBundle\Attribute\AsDoctrineListener;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Mapping\ClassMetadata;
use Doctrine\Persistence\Event\LifecycleEventArgs;
use Gedmo\SoftDeleteable\SoftDeleteable;

#[AsDoctrineListener(event: 'preSoftDelete')]
readonly class SoftDeleteCascadeListener
{
    public function __construct(
        private EntityManagerInterface $entityManager,
    ) {
    }

    public function preSoftDelete(LifecycleEventArgs $args): void
    {
        $entity = $args->getObject();

        if (!$entity instanceof SoftDeleteable) {
            return;
        }

        $metadata = $this->entityManager->getClassMetadata($entity::class);

        foreach ($metadata->associationMappings as $assoc) {
            $field = $assoc['fieldName'];
            $type = $assoc['type'];

            // 🎯 On ignore explicitement les ManyToMany (dangereux à soft-delete)
            if (ClassMetadata::MANY_TO_MANY === $type) {
                continue;
            }

            // 🎯 On ne garde que les relations TO_MANY (ex: OneToMany)
            if (!($type & ClassMetadata::TO_MANY)) {
                continue;
            }

            // 🧼 Génère le getter : getFieldName()
            $getter = 'get'.ucfirst($field);

            if (!method_exists($entity, $getter)) {
                continue; // ⚠️ Getter manquant
            }

            $relatedItems = $entity->$getter();

            if (!is_iterable($relatedItems)) {
                continue; // ⚠️ Le résultat n'est pas itérable
            }

            foreach ($relatedItems as $related) {
                // 🔐 Ne soft-delete que les entités SoftDeleteable
                if ($related instanceof SoftDeleteable) {
                    $this->entityManager->remove($related); // ✅ Déclenche un soft delete
                }
            }
        }
    }
}
