<?php

/*
 * This file is part of the Serenity package.
 *
 * (c) CP Creation <web@cpcreation.fr>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Serenity\CoreBundle\Twig\Components\Form;

use Doctrine\ORM\EntityManagerInterface;
use Serenity\CoreBundle\Notifier\Notification\AlertNotification;
use Serenity\CoreBundle\Service\FlashService;
use Serenity\MediaBundle\Trait\ComponentMediaTrait;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\Form\FormTypeInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\UnprocessableEntityHttpException;
use Symfony\Component\OptionsResolver\Options;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Component\Routing\Exception\RouteNotFoundException;
use Symfony\Component\Routing\RouterInterface;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\Component\Serializer\Normalizer\DenormalizerInterface;
use Symfony\Component\Uid\Uuid;
use Symfony\Contracts\Translation\TranslatorInterface;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveAction;
use Symfony\UX\LiveComponent\Attribute\LiveArg;
use Symfony\UX\LiveComponent\Attribute\LiveListener;
use Symfony\UX\LiveComponent\Attribute\LiveProp;
use Symfony\UX\LiveComponent\ComponentToolsTrait;
use Symfony\UX\LiveComponent\ComponentWithFormTrait;
use Symfony\UX\LiveComponent\DefaultActionTrait;
use Symfony\UX\TwigComponent\Attribute\PreMount;

#[AsLiveComponent(template: '@components/SerenityCore/Form/form.html.twig')]
#[IsGranted('ROLE_MODERATOR')]
class Form
{
    use ComponentMediaTrait;
    use ComponentToolsTrait;
    use ComponentWithFormTrait;
    use DefaultActionTrait;

    /**
     * @var class-string<FormTypeInterface>
     */
    #[LiveProp(writable: false)]
    public string $formType;

    /**
     * @var class-string<object>
     */
    #[LiveProp(writable: false)]
    public string $class;

    #[LiveProp(writable: false)]
    public ?string $messageSuccess = null;

    #[LiveProp(writable: false)]
    public ?string $messageDelete = null;

    #[LiveProp(writable: false)]
    public ?string $redirectNew = null;

    #[LiveProp(writable: false)]
    public ?string $redirectSuccess = null;

    #[LiveProp(writable: false)]
    public ?string $cancelUrl = null;

    #[LiveProp(writable: false)]
    public ?string $cancelAction = null;

    public mixed $data = null;

    #[LiveProp(writable: false)]
    public bool $findOne = false;

    #[LiveProp(writable: false)]
    public bool $newInstance = false;

    #[LiveProp(writable: false)]
    public bool $edit = false;

    #[LiveProp(writable: false)]
    public bool $delete = true;

    /** @var array<string, mixed> */
    #[LiveProp(writable: true)]
    public array $initialFormData = [];

    public function __construct(private readonly EntityManagerInterface $entityManager,
        private FlashService $flash,
        private readonly TranslatorInterface $translator,
        private readonly RouterInterface $router,
        private readonly DenormalizerInterface $serializer,
        private readonly FormFactoryInterface $formFactory,
    ) {
    }

    #[LiveListener('form:refreshForm')]
    public function __invoke(): void
    {
    }

    /**
     * @param array<string> $data
     *
     * @return array<string>
     */
    #[PreMount]
    public function preMount(array $data): array
    {
        $resolver = new OptionsResolver();
        $resolver->setIgnoreUndefined(true);
        $resolver->setRequired('formType');
        $resolver->setRequired('class');
        $resolver->setDefined('messageSuccess');
        $resolver->setDefined('messageDelete');
        $resolver->setDefined('redirectNew');
        $resolver->setDefined('redirectSuccess');
        $resolver->setDefined('cancelUrl');
        $resolver->setDefined('cancelAction');
        $resolver->setDefined('initialFormData');
        $resolver->setDefined('findOne');
        $resolver->setDefined('newInstance');
        $resolver->setDefined('edit');
        $resolver->setDefined('delete');
        $resolver->setDefined('name');
        $resolver->setAllowedTypes('messageSuccess', ['string', 'null']);
        $resolver->setAllowedTypes('messageDelete', ['string', 'null']);
        $resolver->setAllowedTypes('redirectNew', ['string', 'null']);
        $resolver->setAllowedTypes('redirectSuccess', ['string', 'null']);
        $resolver->setAllowedTypes('class', 'string');
        $resolver->setAllowedTypes('cancelUrl', ['string', 'null']);
        $resolver->setAllowedTypes('cancelAction', ['string', 'null']);
        $resolver->setAllowedTypes('initialFormData', ['array', 'null']);
        $resolver->setAllowedTypes('findOne', 'bool');
        $resolver->setAllowedTypes('newInstance', 'bool');
        $resolver->setAllowedTypes('edit', 'bool');
        $resolver->setAllowedTypes('delete', 'bool');
        // Validation custom pour la route :
        $resolver->setNormalizer('cancelUrl', function (Options $options, $value) {
            if (null === $value) {
                return null;
            }

            try {
                $this->router->generate($value);
            } catch (RouteNotFoundException $e) {
                throw new \InvalidArgumentException(\sprintf('The route "%s" does not exist.', $value));
            }

            return $value;
        });

        $resolver->setNormalizer('redirectSuccess', function (Options $options, $value) {
            if (null === $value) {
                return null;
            }

            try {
                $this->router->generate($value);
            } catch (RouteNotFoundException $e) {
                throw new \InvalidArgumentException(\sprintf('The route "%s" does not exist.', $value));
            }

            return $value;
        });

        // Validation custom pour la route :
        $resolver->setNormalizer('redirectNew', function (Options $options, $value) {
            if (null === $value) {
                return null;
            }

            try {
                $this->router->generate($value, ['id' => Uuid::v4()]);
            } catch (RouteNotFoundException $e) {
                throw new \InvalidArgumentException(\sprintf('The route "%s" does not exist.', $value));
            }

            return $value;
        });

        $resolved = $resolver->resolve($data);
        if (($resolved['findOne'] ?? false) && !empty($resolved['initialFormData'])) {
            throw new \InvalidArgumentException('You cannot define both "findOne" and "initialFormData". Only one must be used.');
        }

        $resolver->setNormalizer('delete', function (Options $options, $value) {
            if ($value && null === $options['cancelUrl']) {
                throw new \InvalidArgumentException('The "cancelUrl" option is required when "delete" is true.');
            }

            return $value;
        });

        return $resolved + $data;
    }

    #[LiveAction]
    public function remove(#[LiveArg] Uuid $id): RedirectResponse
    {
        if (null === $this->cancelUrl) {
            throw new \LogicException('Missing cancel URL for delete action.');
        }
        $this->submitForm(false);
        /** @var string $cancelUrl */
        $cancelUrl = $this->cancelUrl;
        $url = $this->router->generate($cancelUrl);
        $entity = $this->entityManager->getRepository($this->class)->find($id);
        if (!$entity && $entity !== $this->getForm()->getData()) {
            return new RedirectResponse($url, 302);
        }
        $this->entityManager->remove($this->getForm()->getData());
        $this->entityManager->flush();

        if ($this->messageDelete) {
            $this->flash->add($this->messageDelete);
        }

        return new RedirectResponse($url, 302);
    }

    protected function instantiateForm(): FormInterface
    {
        $this->data = null;
        if (!empty($this->initialFormData)) {
            $this->data = $this->entityManager->getRepository($this->class)->findOneBy($this->initialFormData);
        }
        if ($this->findOne) {
            $this->data = $this->entityManager->getRepository($this->class)->findOneBy([]);
        }

        if (!$this->data && !empty($this->initialFormData)) {
            try {
                $this->data = $this->serializer->denormalize(
                    $this->initialFormData,
                    $this->class,
                    null, // format (null = utilise les normalizers)
                    ['allow_extra_attributes' => false] // ou true selon besoin
                );
            } catch (\Exception $e) {
                $this->data = null;
            }
        }

        if (null === $this->data && $this->newInstance) {
            $this->data = new $this->class();
        }

        return $this->formFactory->create($this->formType, $this->data, ['edit' => $this->edit]);
    }

    #[LiveAction]
    public function save(Request $request): ?Response
    {
        if ($this->uploadedFiles) {
            $this->formValues = array_replace_recursive($this->formValues, $this->deserializeUploadedFiles($this->uploadedFiles));
        }
        try {
            $this->submitForm();
            if ($this->form instanceof FormInterface) {
                $data = $this->form->getData();
                $this->entityManager->persist($data);
                $this->entityManager->flush();

                if ($this->redirectNew && false === $this->edit) {
                    $url = $this->router->generate($this->redirectNew, ['id' => $data->getId()]);
                    if (null !== $this->messageSuccess) {
                        $this->flash->add($this->messageSuccess);
                    }

                    $this->resetForm();

                    return new RedirectResponse($url, 302);
                }

                if (null !== $this->redirectSuccess) {
                    $url = $this->router->generate($this->redirectSuccess);
                    if (null !== $this->messageSuccess) {
                        $this->flash->add($this->messageSuccess);
                    }
                    $this->resetForm();

                    return new RedirectResponse($url, 302);
                }

                if (null !== $this->messageSuccess) {
                    $this->flash->push($this->messageSuccess);
                }

                $this->uploadedFiles = [];
                $this->resetForm();
            }
        } catch (UnprocessableEntityHttpException $e) {
            $this->entityManager->detach($this->form->getData());
            if ($this->form->getErrors(true)->count() > 0) {
                $this->flash->push($this->translator->trans('form.error', [], 'serenity_core_messages.admin'), AlertNotification::LEVEL_ERROR);
            }
        }

        return null;
    }
}
