<?php

/*
 * This file is part of the Serenity package.
 *
 * (c) CP Creation <web@cpcreation.fr>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Serenity\AdminBundle\Form\Type;

use Serenity\CoreBundle\Contract\SearchInterface;
use Serenity\CoreBundle\Form\Type\MultipleSelectType;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\ButtonType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;

abstract class SearchType extends AbstractType
{
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $searchAttr = array_merge([
            'placeholder' => 'form.search.label',
            'autocomplete' => 'off',
        ], $options['query_attr'] ?? []);
        $builder->add('query', TextType::class, [
            'required' => false,
            'icon_prefix' => 'icon-search',
            'label' => false,
            'translation_domain' => 'serenity_admin_messages',
            'attr' => $searchAttr,
        ]);
        if ($options['sorter']) {
            if (isset($options['sorter_multiple']) && $options['sorter_multiple']) {
                $builder->add('direction', ChoiceType::class, [
                    'choices' => array_combine(SearchInterface::DIRECTION, SearchInterface::DIRECTION),
                    'required' => true,
                    'label' => 'form.direction.label',
                    'translation_domain' => 'serenity_admin_messages',
                    'multiple' => false,
                    'data' => 'ASC',
                    'expanded' => true,
                ]);

                $builder->add('sorter', MultipleSelectType::class, [
                    'choices' => $this->getSorterChoices($options['sorter_choices']),
                    'translation_domain' => 'serenity_search.admin',
                    'required' => $options['sorter_required'],
                    'attr' => $options['sorter_attr'] ?? [],
                    'multiple' => true,
                    'label' => false,
                ]);
            } else {
                $builder->add('sorter', ChoiceType::class, [
                    'choices' => $this->getSorterChoices($options['sorter_choices']),
                    'translation_domain' => 'serenity_search.admin',
                    'required' => $options['sorter_required'],
                    'attr' => $options['sorter_attr'] ?? [],
                    'multiple' => false,
                    'label' => false,
                ]);
            }
        }

        if (isset($options['button_submit']) && $options['button_submit']) {
            $builder->add('submit', ButtonType::class, [
                'label' => 'Filter',
                'translation_domain' => 'serenity_search.admin',
                'attr' => [
                    'data-action' => 'live#action',
                    'data-live-action-param' => 'debounce(300)|filter',
                ],
            ]);
        }
    }

    /**
     * @param array<string, string|array{property?: string, join?: string, method?: string}> $sorterConfig
     *
     * @return array<string, string>
     */
    protected function getSorterChoices(array $sorterConfig): array
    {
        $choices = [];

        foreach ($sorterConfig as $label => $config) {
            if (\is_string($config)) {
                // Cas simple : champ direct
                $choices[$label] = $label;
            } elseif (\is_array($config)) {
                $choices[$label] = $label;
            }
        }

        return $choices;
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => SearchInterface::class,
            'csrf_protection' => false,
            'query_attr' => [],
            'sorter' => false,
            'sorter_choices' => SearchInterface::SORTER,
            'sorter_multiple' => false,
            'sorter_attr' => [],
            'button_submit' => false,
            'sorter_required' => false,
        ]);

        $resolver->setAllowedTypes('query_attr', 'array');
        $resolver->setAllowedTypes('sorter_attr', 'array');
    }
}
